#!/usr/bin/env bash

##########
# This script will flash a The Fairphone (Gen. 6)
# Depending on the toggles, different operations can be performed.
##########

set -e
set -u

# Colors
GREEN='\033[1;32m'
YELLOW='\033[1;33m'
RED='\033[1;31m'
NC='\033[0m' # No Color

# Toggles
AUTO_REBOOT="true"           # Control reboot behavior. Default is "true"
DATA_WIPE="true"             # User data wipe. Default is "true"
DRY_RUN="false"              # No-flash run. Default value is "false"
FRP_WIPE="true"              # frp wipe. Default is "true"
INTEGRITY_CHECK="true"       # Checksum run. default value is "true"
REBOOT_TO_BOOTLOADER="false" # default value is "false"

# Target device info
PRODUCT="The Fairphone (Gen. 6)"
PRODUCT_ID="FP6"

# Software release info
PACKAGE_TYPE="\"user-factory\""

# Paths/files
ROOT_DIR="$( cd -- "$( dirname -- "${BASH_SOURCE[0]}" )" &> /dev/null && pwd )"
IMAGES_DIR="${ROOT_DIR}/images"
SHA256SUMS="SHA256SUMS" # Path to file not needed, only the filename

# Abort the script (and wait for a key to be pressed).
abort_now() {
    echo -en "$RED"
    read -rp "ERROR: Aborting now (press Enter to terminate)." a
    echo -en "$NC"
    echo ""
    exit 1
}

# Warn about data wipe, and ask for confirmation
data_wipe_check() {
    # Adapt the warning based on the value of the FRP_WIPE toggle
    if [ "${FRP_WIPE}" != "true" ]; then
        echo -en "$YELLOW"
        echo "WARNING: Flashing this image wipes all user data and settings on the phone."
        echo "         Are you sure you want to continue?"
    elif [ "${FRP_WIPE}" = "true" ]; then
        echo -en "$YELLOW"
        echo "WARNING: Flashing this image wipes all user data and settings on the phone."
        echo "         It will also remove the Google factory reset protection."
        echo "         Are you sure you want to continue?"
    fi
    # Read user's input
    read -rp "         Type \"Yes\" (case sensitive) and press enter to continue: " a
    echo -en "$NC"
    if [ "_${a:-"No"}" != '_Yes' ]; then # NOTE: $a is being set by the read command above,
        #       so the check for a to be set is mostly redundant
        echo -en "$YELLOW"
        echo "WARNING: You DID NOT type \"Yes\", exiting now."
        echo -en "$NC"
        exit 1
    fi
}

# Check for connected phone
find_device() {
    echo "INFO: Looking for connected device(s)..."
    DEVICE_FOUND="false"
    while [ ${DEVICE_FOUND} = "false" ]; do
        serial_numbers=

        for sn in $("${FASTBOOT_BIN}" devices | grep fastboot | grep -oE '^[[:alnum:]]+'); do
            # Checking the product ID
            PRODUCT_STRING=$("${FASTBOOT_BIN}" -s "${sn}" getvar product 2>&1)
            # Add serial, if product matches
            if [[ ${PRODUCT_STRING} == *"${PRODUCT_ID}"* ]]; then
                serial_numbers="${serial_numbers} ${sn}"
                ANDROID_SERIAL="${sn}"
            fi
        done

        case $(echo "${serial_numbers}" | wc -w | grep -oE '[0-9]+') in
            0)
                echo -en "$YELLOW"
                echo "WARNING: No ${PRODUCT} found in fastboot mode."
                echo "WARNING: Make sure that a ${PRODUCT} is connected."
                ;;
            1)
                echo "INFO: One ${PRODUCT} in fastboot mode found (serial number: ${ANDROID_SERIAL})."

                DEVICE_FOUND="true"
                break
                ;;
            *)
                echo -en "$YELLOW"
                echo "WARNING: Several ${PRODUCT}'s in fastboot mode connected."
                echo "WARNING: Please connect only one ${PRODUCT}."
                ;;
        esac

        while true; do
            read -rp "         Do you want to look for a ${PRODUCT} again? [(Y)es/(n)o]: " a
            echo -en "$NC"
            if [ -z "${a}" ] || [ "${a}" = 'y' ] || [ "${a}" = 'Y' ]; then
                break
            elif [ "${a}" = 'n' ] || [ "${a}" = 'N' ]; then
                exit 0
            fi
        done
    done
}

# Flashing and other operations
flash_device() {

    echo "INFO: flashing partitions..."

    flash_image_or_abort abl_a "${IMAGES_DIR}/abl.elf"
    flash_image_or_abort abl_b "${IMAGES_DIR}/abl.elf"
    flash_image_or_abort aop_a "${IMAGES_DIR}/aop.mbn"
    flash_image_or_abort aop_b "${IMAGES_DIR}/aop.mbn"
    flash_image_or_abort aop_config_a "${IMAGES_DIR}/aop_devcfg.mbn"
    flash_image_or_abort aop_config_b "${IMAGES_DIR}/aop_devcfg.mbn"
    flash_image_or_abort bluetooth_a "${IMAGES_DIR}/BTFM.bin"
    flash_image_or_abort bluetooth_b "${IMAGES_DIR}/BTFM.bin"
    flash_image_or_abort cpucp_a "${IMAGES_DIR}/cpucp.elf"
    flash_image_or_abort cpucp_b "${IMAGES_DIR}/cpucp.elf"
    flash_image_or_abort cpucp_dtb_a "${IMAGES_DIR}/cpucp_dtbs.elf"
    flash_image_or_abort cpucp_dtb_b "${IMAGES_DIR}/cpucp_dtbs.elf"
    flash_image_or_abort devcfg_a "${IMAGES_DIR}/devcfg.mbn"
    flash_image_or_abort devcfg_b "${IMAGES_DIR}/devcfg.mbn"
    flash_image_or_abort dsp_a "${IMAGES_DIR}/dspso.bin"
    flash_image_or_abort dsp_b "${IMAGES_DIR}/dspso.bin"
    flash_image_or_abort featenabler_a "${IMAGES_DIR}/featenabler.mbn"
    flash_image_or_abort featenabler_b "${IMAGES_DIR}/featenabler.mbn"
    flash_image_or_abort hyp_a "${IMAGES_DIR}/hypvm.mbn"
    flash_image_or_abort hyp_b "${IMAGES_DIR}/hypvm.mbn"
    flash_image_or_abort imagefv_a "${IMAGES_DIR}/imagefv.elf"
    flash_image_or_abort imagefv_b "${IMAGES_DIR}/imagefv.elf"
    flash_image_or_abort keymaster_a "${IMAGES_DIR}/keymint.mbn"
    flash_image_or_abort keymaster_b "${IMAGES_DIR}/keymint.mbn"
    flash_image_or_abort logfs "${IMAGES_DIR}/logfs_ufs_8mb.bin"
    flash_image_or_abort modem_a "${IMAGES_DIR}/NON-HLOS.bin"
    flash_image_or_abort modem_b "${IMAGES_DIR}/NON-HLOS.bin"
    flash_image_or_abort multiimgoem_a "${IMAGES_DIR}/multi_image.mbn"
    flash_image_or_abort multiimgoem_b "${IMAGES_DIR}/multi_image.mbn"
    flash_image_or_abort pvmfw_a "${IMAGES_DIR}/pvmfw.img"
    flash_image_or_abort pvmfw_b "${IMAGES_DIR}/pvmfw.img"
    flash_image_or_abort qupfw_a "${IMAGES_DIR}/qupv3fw.elf"
    flash_image_or_abort qupfw_b "${IMAGES_DIR}/qupv3fw.elf"
    flash_image_or_abort shrm_a "${IMAGES_DIR}/shrm.elf"
    flash_image_or_abort shrm_b "${IMAGES_DIR}/shrm.elf"
    flash_image_or_abort storsec "${IMAGES_DIR}/storsec.mbn"
    flash_image_or_abort study "${IMAGES_DIR}/study.img"
    flash_image_or_abort studybk_a "${IMAGES_DIR}/study.img"
    flash_image_or_abort studybk_b "${IMAGES_DIR}/study.img"
    flash_image_or_abort toolsfv "${IMAGES_DIR}/tools.fv"
    flash_image_or_abort tz_a "${IMAGES_DIR}/tz.mbn"
    flash_image_or_abort tz_b "${IMAGES_DIR}/tz.mbn"
    flash_image_or_abort uefi_a "${IMAGES_DIR}/uefi.elf"
    flash_image_or_abort uefi_b "${IMAGES_DIR}/uefi.elf"
    flash_image_or_abort uefisecapp_a "${IMAGES_DIR}/uefi_sec.mbn"
    flash_image_or_abort uefisecapp_b "${IMAGES_DIR}/uefi_sec.mbn"
    flash_image_or_abort vm-bootsys_a "${IMAGES_DIR}/vm-bootsys.img"
    flash_image_or_abort vm-bootsys_b "${IMAGES_DIR}/vm-bootsys.img"
    flash_image_or_abort vm-persist "${IMAGES_DIR}/vm-persist.img"
    flash_image_or_abort xbl_a "${IMAGES_DIR}/xbl_s.melf"
    flash_image_or_abort xbl_b "${IMAGES_DIR}/xbl_s.melf"
    flash_image_or_abort xbl_config_a "${IMAGES_DIR}/xbl_config.elf"
    flash_image_or_abort xbl_config_b "${IMAGES_DIR}/xbl_config.elf"
    flash_image_or_abort xbl_ramdump_a "${IMAGES_DIR}/XblRamdump.elf"
    flash_image_or_abort xbl_ramdump_b "${IMAGES_DIR}/XblRamdump.elf"

    flash_image_or_abort boot_a "${IMAGES_DIR}/boot.img"
    flash_image_or_abort boot_b "${IMAGES_DIR}/boot.img"
    flash_image_or_abort dtbo_a "${IMAGES_DIR}/dtbo.img"
    flash_image_or_abort dtbo_b "${IMAGES_DIR}/dtbo.img"
    flash_image_or_abort init_boot_a "${IMAGES_DIR}/init_boot.img"
    flash_image_or_abort init_boot_b "${IMAGES_DIR}/init_boot.img"
    flash_image_or_abort recovery_a "${IMAGES_DIR}/recovery.img"
    flash_image_or_abort recovery_b "${IMAGES_DIR}/recovery.img"
    flash_image_or_abort super "${IMAGES_DIR}/super.img"
    flash_image_or_abort vbmeta_a "${IMAGES_DIR}/vbmeta.img"
    flash_image_or_abort vbmeta_b "${IMAGES_DIR}/vbmeta.img"
    flash_image_or_abort vbmeta_system_a "${IMAGES_DIR}/vbmeta_system.img"
    flash_image_or_abort vbmeta_system_b "${IMAGES_DIR}/vbmeta_system.img"
    flash_image_or_abort vendor_boot_a "${IMAGES_DIR}/vendor_boot.img"
    flash_image_or_abort vendor_boot_b "${IMAGES_DIR}/vendor_boot.img"


    # Process partitions that must be handled for a full wipe
    if [ "${DATA_WIPE}" = "true" ]; then
        echo "INFO: Deleting user data"
        flash_image_or_abort userdata "${IMAGES_DIR}/userdata.img"
        flash_image_or_abort metadata "${IMAGES_DIR}/metadata.img"
    fi

    # Process partitions with FRP-related data
    if [ "${FRP_WIPE}" = "true" ]; then
        echo "INFO: Deleting factory reset partition"
        flash_image_or_abort frp "${IMAGES_DIR}/frp_for_factory.img"

    fi

    echo "INFO: Erasing some partitions"
    "${FASTBOOT_BIN}" -s "${ANDROID_SERIAL}" erase misc
    "${FASTBOOT_BIN}" -s "${ANDROID_SERIAL}" erase modemst1
    "${FASTBOOT_BIN}" -s "${ANDROID_SERIAL}" erase modemst2

    echo "INFO: Activating partition slot A"
    "${FASTBOOT_BIN}" -s "${ANDROID_SERIAL}" --set-active=a
}

# Flash an image to a partition. Abort on failure.
# Arguments: <partition name> <image file>
flash_image_or_abort() {
    local partition=$1
    local image_file=$2
    ("${FASTBOOT_BIN}" -s "${ANDROID_SERIAL}" flash "${partition}" "${image_file}") ||
        {
            echo -en "$RED"
            echo "ERROR: Could not flash the ${partition} partition on device ${ANDROID_SERIAL}."
            echo ""
            echo "ERROR: Please unplug the phone, take the battery out, boot the device into"
            echo "ERROR: fastboot mode, and start this script again."
            echo "ERROR: (To get to fastboot mode, press Volume-Down and plug in the USB-C)"
            echo "ERROR: (cable until the fastboot menu appears.)"
            abort_now
        }
}

# Check if the device is unlocked. If not, stop the script.
is_unlocked() {
    # Store the string that we will look for when checking
    UNLOCKED_STRING="Device unlocked: true"
    CRITICAL_UNLOCKED_STRING="Device critical unlocked: true"

    # Get the current status of the bootloader
    # Note that the output goes to stderr, so we need a redirect
    DEVICE_INFO_OUTPUT=$("${FASTBOOT_BIN}" -s "${ANDROID_SERIAL}" oem device-info 2>&1)

    # Check our strings against the output of the status command.
    # Provide feedback to the user, and exit if needed.
    if [[ $DEVICE_INFO_OUTPUT == *$UNLOCKED_STRING* ]]; then
        if [[ $DEVICE_INFO_OUTPUT == *$CRITICAL_UNLOCKED_STRING* ]]; then
            echo "INFO: The device is fully unlocked."
        else
            echo -en "$RED"
            echo "ERROR: The critical partitions are not unlocked."
            echo "       Please refer to our support articles for help."
            abort_now
        fi
    else
        echo -en "$RED"
        echo "ERROR: The device is not unlocked."
        echo "       Please refer to our support articles for help."
        abort_now
    fi
}

# Operating system checks and variable definition
os_checks() {
    case "$(uname -s 2>/dev/null)" in
        Darwin)
            echo "INFO: You are using macOS."
            FASTBOOT_BIN="${ROOT_DIR}/bin-darwin/fastboot"
            SHA256SUM_BIN=$(which shasum)
            SHA256SUM_CMD=("${SHA256SUM_BIN}" --algorithm 256 --status --check "${SHA256SUMS}")
            ;;
        Linux|GNU/Linux)
            echo "INFO: You are using a Linux distribution."
            # Find fastboot, abort if not available.
            FASTBOOT_BIN=$(which fastboot) || {
                echo -en "$RED"
                echo "ERROR: The 'fastboot' tool is not available."
                echo "ERROR: Please install the tool using your package manager."
                abort_now
            }
            SHA256SUM_BIN=$(which sha256sum)
            SHA256SUM_CMD=("${SHA256SUM_BIN}" --status --check "${SHA256SUMS}")
            ;;
        msys|MINGW*)
            echo "INFO: You are using MinGW on Windows"
            FASTBOOT_BIN="${ROOT_DIR}/bin-msys/fastboot.exe"
            SHA256SUM_BIN=$(which sha256sum)
            SHA256SUM_CMD=("${SHA256SUM_BIN}" --status --check "${SHA256SUMS}")
            ;;
        *)
            echo -en "$RED"
            echo "ERROR: Unsupported operating system (${OSTYPE})."
            echo "ERROR: Only GNU/Linux, macOS, and MinGW on Windows are supported."
            abort_now
            ;;
    esac
}

# Validate checksums. Abort on failure.
# Arguments: <validation command line...>
validate_checksums() {
    echo ""
    echo "INFO: Validating the integrity of the package."
    echo "      This might take a while. Please wait..."
    # <validation command> must run within the correct folder, so we cd into it.
    # This is because the digest file used by the command uses relative paths.
    # Absolute paths are not an option, as they would break portability.
    cd "$ROOT_DIR"
    ("${@}") ||
        {
            echo -en "$RED"
            echo "ERROR: Checksums do not match."
            echo "ERROR: Please download the package again."
            abort_now
        }

    echo "INFO: Validation complete."
}

integrity_check() {
    if [ "${INTEGRITY_CHECK}" = "false" ]; then
        echo -en "$YELLOW"
        echo "WARNING: Skipping file validation check, as per hard-coded toggle"
        echo -en "$NC"
        return
    fi

    # If the integrity check is requested, we look for the binary.
    if [ -x "${SHA256SUM_BIN}" ]; then
        # With the binary found, we start the check.
        validate_checksums "${SHA256SUM_CMD[@]}"
    else
        echo ""
        echo "INFO: No SHA-256 checksum program is available."
        echo "INFO: Will proceed anyway."
        echo ""
    fi
}

# Control the reboot sequence
reboot_device() {
    echo "-----------"
    echo ""
    if [ ${AUTO_REBOOT} == "false" ]; then
        # Listen for user input
        read -rp "INFO: Device flashed. Press any key to reboot..." a
    else
        echo "INFO: Device flashed. Rebooting now..."
    fi
    # Reboot the device. Target state defined by toggle.
    if [ "$REBOOT_TO_BOOTLOADER" = "true" ]; then
        "${FASTBOOT_BIN}" -s "${ANDROID_SERIAL}" reboot bootloader
    else
        "${FASTBOOT_BIN}" -s "${ANDROID_SERIAL}" reboot
    fi
    echo -en "$GREEN"
    echo ""
    echo "INFO: ************************"
    echo "      * OPERATION SUCCESSFUL *"
    echo "      ************************"
    echo ""
    echo "INFO: You can unplug the USB cable now."
    echo "INFO: This script will terminate in 10 seconds. Bye! :)"
    echo -en "$NC"
    echo ""
    sleep 10
}


echo ""
echo "*** ${PRODUCT} flashing script ***"
echo ""
echo "INFO: The procedure will start soon. Please wait..."
echo "INFO: The package type is ${PACKAGE_TYPE}"

# Begin with some OS checks and variable definition
os_checks

# Just some buffer
sleep 1

# Run integrity check
integrity_check

# Warn about data wipe, and ask for confirmation
if [ "${DATA_WIPE}" = "true" ]; then
    data_wipe_check
fi

# Call function to look for device(s)
find_device

# Make sure the device is unlocked
is_unlocked

# Turn into a dry run
if [ "${DRY_RUN}" = "true" ]; then
    echo -en "$YELLOW"
    echo "WARNING: Dry run mode enabled. Not flashing..."
    echo -en "$NC"
else
    # Call flashing function
    echo "INFO: Proceeding to flash the device."
    flash_device
fi

reboot_device
